<?php
declare(strict_types=1);

function table_exists(string $table): bool {
  $row = DB::one("SELECT name FROM sqlite_master WHERE type='table' AND name=?", [$table]);
  return (bool)$row;
}

function column_exists(string $table, string $column): bool {
  $rows = DB::all('PRAGMA table_info(' . $table . ')');
  foreach ($rows as $r) { if (strcasecmp((string)($r['name'] ?? ''), $column) === 0) return true; }
  return false;
}

function ensure_touch_trigger(string $table): void {
  $trigger = $table . '_updated_at_touch';
  DB::exec("DROP TRIGGER IF EXISTS {$trigger}");
  DB::exec("CREATE TRIGGER IF NOT EXISTS {$trigger} AFTER UPDATE ON {$table}
            BEGIN
              UPDATE {$table} SET updated_at = CURRENT_TIMESTAMP WHERE rowid = NEW.rowid;
            END;");
}

function run_migrations(): void {
  if (!file_exists(DB_PATH)) @touch(DB_PATH);
  DB::exec('CREATE TABLE IF NOT EXISTS schema_version (version INTEGER NOT NULL)');
  $row = DB::one('SELECT version FROM schema_version LIMIT 1');
  $ver = $row ? (int)$row['version'] : 0;

  /* v1: keep auth */
  if ($ver < 1) {
    DB::exec('CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      username TEXT UNIQUE,
      password_hash TEXT,
      must_change INTEGER DEFAULT 1,
      created_at TEXT
    )');
    $u = DB::one('SELECT COUNT(*) AS c FROM users');
    if (!$u || (int)$u['c'] === 0) {
      $hash = password_hash("admin123", PASSWORD_BCRYPT);
      DB::exec('INSERT INTO users (username,password_hash,must_change,created_at) VALUES (?,?,1,?)',
               ['admin', $hash, gmdate('c')]);
    }
    if ($row) DB::exec('UPDATE schema_version SET version=1'); else DB::exec('INSERT INTO schema_version (version) VALUES (1)');
    $ver = 1;
  }

  /* v2: NIO core schema (use CURRENT_TIMESTAMP defaults) */
  if ($ver < 2) {
    DB::exec('CREATE TABLE IF NOT EXISTS nio_services (
      id          INTEGER PRIMARY KEY AUTOINCREMENT,
      name        TEXT NOT NULL,
      url         TEXT NOT NULL DEFAULT "",
      status      TEXT NOT NULL DEFAULT "active",
      created_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    )');
    ensure_touch_trigger('nio_services');
    DB::exec('CREATE INDEX IF NOT EXISTS idx_nio_services_name ON nio_services(name)');
    DB::exec('CREATE INDEX IF NOT EXISTS idx_nio_services_status ON nio_services(status)');

    DB::exec('CREATE TABLE IF NOT EXISTS nio_active_codes (
      id          INTEGER PRIMARY KEY AUTOINCREMENT,
      code        TEXT NOT NULL UNIQUE,
      service_id  INTEGER NULL REFERENCES nio_services(id) ON DELETE SET NULL,
      username    TEXT DEFAULT "",
      password    TEXT DEFAULT "",
      status      TEXT NOT NULL DEFAULT "active",
      notes       TEXT,
      created_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    )');
    ensure_touch_trigger('nio_active_codes');
    DB::exec('CREATE INDEX IF NOT EXISTS idx_nio_codes_status ON nio_active_codes(status)');
    DB::exec('CREATE INDEX IF NOT EXISTS idx_nio_codes_service ON nio_active_codes(service_id)');

    DB::exec('CREATE TABLE IF NOT EXISTS nio_settings (
      k           TEXT PRIMARY KEY,
      v           TEXT,
      created_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    )');
    ensure_touch_trigger('nio_settings');

    // Seed defaults (broader compatibility than UPSERT)
    foreach ([
      'logo_url'       => '',
      'background_url' => '',
      'app_mode'       => 'Xtream',
      'privacy_policy' => 'https://pastebin.com/raw/JiimGEjk',
      'legal_msg'      => '',
      'sports'         => '',
    ] as $k => $v) {
      DB::exec('INSERT OR IGNORE INTO nio_settings (k, v) VALUES (?, ?)', [$k, $v]);
    }

    DB::exec('CREATE TABLE IF NOT EXISTS nio_announcements (
      id          INTEGER PRIMARY KEY AUTOINCREMENT,
      title       TEXT NOT NULL,
      message     TEXT NOT NULL,
      image       TEXT NOT NULL DEFAULT "",
      status      TEXT NOT NULL DEFAULT "active",
      created_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    )');
    ensure_touch_trigger('nio_announcements');
    DB::exec('CREATE INDEX IF NOT EXISTS idx_nio_ann_status ON nio_announcements(status)');

    DB::exec('CREATE TABLE IF NOT EXISTS nio_notifications (
      id          INTEGER PRIMARY KEY AUTOINCREMENT,
      title       TEXT NOT NULL,
      message     TEXT NOT NULL,
      status      TEXT NOT NULL DEFAULT "active",
      created_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at  TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    )');
    ensure_touch_trigger('nio_notifications');
    DB::exec('CREATE INDEX IF NOT EXISTS idx_nio_notif_status ON nio_notifications(status)');

    DB::exec('UPDATE schema_version SET version=2'); $ver = 2;
  }

  /* v3: repairs */
  if ($ver < 3) {
    if (table_exists('nio_announcements') && !column_exists('nio_announcements','image')) {
      DB::exec('ALTER TABLE nio_announcements ADD COLUMN image TEXT NOT NULL DEFAULT ""');
    }
    if (table_exists('nio_active_codes')) {
      if (!column_exists('nio_active_codes','username')) {
        DB::exec('ALTER TABLE nio_active_codes ADD COLUMN username TEXT DEFAULT ""');
      }
      if (!column_exists('nio_active_codes','password')) {
        DB::exec('ALTER TABLE nio_active_codes ADD COLUMN password TEXT DEFAULT ""');
      }
    }
    DB::exec("UPDATE nio_settings SET v='' WHERE k IN ('logo_url','background_url','legal_msg','sports') AND (v IS NULL OR v='' OR v='false')");
    DB::exec("UPDATE nio_settings SET v='Xtream' WHERE k='app_mode' AND (v IS NULL OR v='' OR v='false')");
    DB::exec("UPDATE nio_settings SET v='https://pastebin.com/raw/JiimGEjk' WHERE k='privacy_policy' AND (v IS NULL OR v='' OR v='false')");

    DB::exec('UPDATE schema_version SET version=3'); $ver = 3;
  }

  /* v4: webview_pages (keep & fully index; CURRENT_TIMESTAMP defaults) */
  if ($ver < 4) {
    $hasTable = table_exists('webview_pages');

    DB::exec('BEGIN');
    try {
      if (!$hasTable) {
        DB::exec('CREATE TABLE IF NOT EXISTS webview_pages (
          id INTEGER PRIMARY KEY AUTOINCREMENT,
          title TEXT,
          url TEXT,
          active INTEGER DEFAULT 1,
          notes TEXT,
          created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
          updated_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
        )');
      } else {
        $cols = DB::all('PRAGMA table_info(webview_pages)');
        $have = [];
        foreach ($cols as $c) { $have[strtolower((string)$c['name'])] = true; }

        DB::exec('CREATE TABLE webview_pages__new (
          id INTEGER PRIMARY KEY AUTOINCREMENT,
          title TEXT,
          url TEXT,
          active INTEGER DEFAULT 1,
          notes TEXT,
          created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
          updated_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
        )');

        $hasId      = !empty($have['id']);
        $hasTitle   = !empty($have['title']);
        $hasUrl     = !empty($have['url']);
        $hasActive  = !empty($have['active']);
        $hasNotes   = !empty($have['notes']);
        $hasCreated = !empty($have['created_at']);
        $hasUpdated = !empty($have['updated_at']);

        $targetCols = [];
        $selectExpr = [];

        if ($hasId) { $targetCols[] = 'id'; $selectExpr[] = 'id'; }
        $targetCols[] = 'title';       $selectExpr[] = $hasTitle  ? 'title'  : "'' AS title";
        $targetCols[] = 'url';         $selectExpr[] = $hasUrl    ? 'url'    : "'' AS url";
        $targetCols[] = 'active';      $selectExpr[] = $hasActive ? 'active' : '1 AS active';
        $targetCols[] = 'notes';       $selectExpr[] = $hasNotes  ? 'notes'  : "'' AS notes";
        $targetCols[] = 'created_at';  $selectExpr[] = $hasCreated? 'created_at' : 'CURRENT_TIMESTAMP AS created_at';
        $targetCols[] = 'updated_at';  $selectExpr[] = $hasUpdated? 'updated_at' : 'CURRENT_TIMESTAMP AS updated_at';

        $insCols = implode(', ', $targetCols);
        $selCols = implode(', ', $selectExpr);

        DB::exec("INSERT INTO webview_pages__new ({$insCols}) SELECT {$selCols} FROM webview_pages");
        DB::exec('DROP TABLE webview_pages');
        DB::exec('ALTER TABLE webview_pages__new RENAME TO webview_pages');
      }

      DB::exec('CREATE INDEX IF NOT EXISTS idx_webview_active ON webview_pages(active)');
      DB::exec('CREATE INDEX IF NOT EXISTS idx_webview_created ON webview_pages(created_at)');
      ensure_touch_trigger('webview_pages');

      DB::exec('UPDATE schema_version SET version=4');
      $ver = 4;
      DB::exec('COMMIT');
    } catch (Throwable $e) {
      DB::exec('ROLLBACK');
      throw $e;
    }
  }
}
