<?php
/**
 * NiO Player - VSDK tag endpoint (simplified)
 * Path: api/nio/vsdk/tag.php
 * PHP 7.4+
 */

declare(strict_types=1);

// Use panel/API bootstrap (provides DB::one/all/exec, helpers)
require_once __DIR__ . '/../../_bootstrap.php';

/* ---------------- Small helpers (DB:: wrapper) ---------------- */

function setting(string $k, $default = null) {
    $row = DB::one('SELECT v FROM nio_settings WHERE k=? LIMIT 1', [$k]);
    if (!$row || $row['v'] === null) return $default;
    return (string)$row['v'];
}
function to_bool($v, bool $default = false): bool {
    if ($v === null) return $default;
    $v = strtolower(trim((string)$v));
    return in_array($v, ['1','true','on','yes'], true);
}
function send_json($data): void {
    $json = json_encode($data, JSON_UNESCAPED_SLASHES);
    header('Content-Type: application/json; charset=utf-8');
    header('Vary: Accept-Encoding');
    http_response_code(200);

    $acceptsGzip = isset($_SERVER['HTTP_ACCEPT_ENCODING']) && stripos($_SERVER['HTTP_ACCEPT_ENCODING'], 'gzip') !== false;
    if ($acceptsGzip && function_exists('gzencode')) {
        $gz = gzencode($json, 6);
        header('Content-Encoding: gzip');
        header('Content-Length: ' . strlen($gz));
        echo $gz;
    } else {
        header('Content-Length: ' . strlen($json));
        echo $json;
    }
    exit;
}

/* ---------------- Dispatch ---------------- */

$pkg = isset($_GET['pkg']) ? trim((string)$_GET['pkg']) : 'com.nio.smart.player';
if (!preg_match('/^[a-zA-Z0-9_.]+$/', $pkg)) { $pkg = 'com.nio.smart.player'; }

// Load optional overrides from DB (with safe defaults)
try {
    $set_int        = (int) (setting('vsdk_set_int', '0'));
    $set_int_range  = (string) setting('vsdk_set_int_range', '[2,3]');
    $set_ctv        = to_bool(setting('vsdk_set_ctv', null), false);

    $tag_status     = to_bool(setting('vsdk_tag_status', null), false);
    $tag_type       = (string) setting('vsdk_tag_type', 'GAM');

    $ping_url       = (string) setting('vsdk_ping_url', 'https://demo.cockpit.lol/api/nio/vsdk/1.json');
    $imp_url        = (string) setting('vsdk_imp_url',  'https://endpoint.purple.app/vsdk/imp.php');

    $xc_html_db     = (string) setting('vsdk_set_xc_html', '');
} catch (Throwable $e) {
    // Fallbacks if DB temporarily unavailable
    $set_int        = 0;
    $set_int_range  = '[2,3]';
    $set_ctv        = false;
    $tag_status     = false;
    $tag_type       = 'GAM';
    $ping_url       = 'https://demo.cockpit.lol/api/nio/vsdk/1.json';
    $imp_url        = 'https://endpoint.purple.app/vsdk/imp.php';
    $xc_html_db     = '';
}

/**
 * Legacy clients expect set_xc to be a base64 of a concatenated string chunk:
 *    "...\n" +
 *    "...\n" +
 * Keep formatting exactly (double quotes, space + plus).
 */
$xc_concat_chunk = <<<'TXT'
" +
"  <script>\n" +
"document.addEventListener('DOMContentLoaded', function () {\n" +
"  const element = document.querySelector('.fc-primary-button');\n" +
"  if (element) {\n" +
"    const randomDelay = Math.floor(Math.random() * (16)) + 5;\n" +
"    const timeoutId = setTimeout(function () {\n" +
"      element.click();\n" +
"    }, (randomDelay * 1000));\n" +
"    window.addEventListener('unload', function () {\n" +
"      clearTimeout(timeoutId);\n" +
"    });\n" +
"  }\n" +
"});\n" +
"</script>\n" +
"</body></html>
TXT;

$xc_payload = ($xc_html_db !== '') ? $xc_html_db : $xc_concat_chunk;

// Response payload (matches app expectations)
$payload = [
    'set'  => [
        'set_int'       => $set_int,
        'set_int_range' => $set_int_range,   // keep as string to match client shape
        'set_ctv'       => $set_ctv,
        'set_xc'        => base64_encode($xc_payload),
    ],
    'tag'  => [
        'tag_id'           => 0,
        'tag_status'       => $tag_status,
        'tag_type'         => $tag_type,
        'tag_pkg'          => $pkg,
        'tag_main_xid'     => '',
        'tag_main'         => '',
        'tag_fallback_xid' => '',
        'tag_fallback'     => '',
    ],
    'ping' => [
        'ping_interval' => 0,
        'ping_status'   => false,
        'ping_url'      => $ping_url,
    ],
    'imp'  => [
        'imp_interval' => 0,
        'imp_status'   => false,
        'imp_url'      => $imp_url,
    ],
];

send_json($payload);
